/**
 * 打包脚本 - 将项目打包到dist文件夹并进行代码压缩
 */

const fs = require("fs");
const path = require("path");
const { execSync } = require("child_process");

// 确保dist目录存在，如果存在则清空
function ensureDistDirectory() {
  const distPath = path.join(__dirname, "dist");

  if (fs.existsSync(distPath)) {
    console.log("清空dist目录...");
    fs.rmSync(distPath, { recursive: true, force: true });
  }

  console.log("创建dist目录...");
  fs.mkdirSync(distPath, { recursive: true });

  return distPath;
}

// 复制并压缩文件
function copyAndMinify(sourcePath, targetPath, options = {}) {
  // 如果源路径是目录，则递归复制
  if (fs.statSync(sourcePath).isDirectory()) {
    if (!fs.existsSync(targetPath)) {
      fs.mkdirSync(targetPath, { recursive: true });
    }

    const files = fs.readdirSync(sourcePath);
    for (const file of files) {
      const srcFile = path.join(sourcePath, file);
      const destFile = path.join(targetPath, file);
      copyAndMinify(srcFile, destFile, options);
    }
    return;
  }

  // 处理文件
  const ext = path.extname(sourcePath).toLowerCase();

  // 创建目标文件的目录（如果不存在）
  const targetDir = path.dirname(targetPath);
  if (!fs.existsSync(targetDir)) {
    fs.mkdirSync(targetDir, { recursive: true });
  }

  // 根据文件类型进行不同处理
  switch (ext) {
    case ".js":
      minifyJS(sourcePath, targetPath);
      break;
    case ".css":
      minifyCSS(sourcePath, targetPath);
      break;
    case ".html":
      minifyHTML(sourcePath, targetPath);
      break;
    case ".json":
      minifyJSON(sourcePath, targetPath);
      break;
    default:
      // 直接复制其他类型的文件
      fs.copyFileSync(sourcePath, targetPath);
      break;
  }
}

// 压缩JS文件
function minifyJS(sourcePath, targetPath) {
  try {
    console.log(`压缩JS: ${path.relative(__dirname, sourcePath)}`);

    // 使用terser压缩JS
    // 如果没有安装terser，先安装
    try {
      require.resolve("terser");
    } catch (e) {
      console.log("安装terser...");
      execSync("npm install terser --no-save");
    }

    const terser = require("terser");
    const code = fs.readFileSync(sourcePath, "utf8");

    terser
      .minify(code, {
        compress: true,
        mangle: true,
      })
      .then((result) => {
        fs.writeFileSync(targetPath, result.code);
      })
      .catch((err) => {
        console.error(`压缩JS文件失败: ${sourcePath}`, err);
        // 如果压缩失败，直接复制原文件
        fs.copyFileSync(sourcePath, targetPath);
      });
  } catch (error) {
    console.error(`处理JS文件失败: ${sourcePath}`, error);
    // 出错时直接复制原文件
    fs.copyFileSync(sourcePath, targetPath);
  }
}

// 压缩CSS文件
function minifyCSS(sourcePath, targetPath) {
  try {
    console.log(`压缩CSS: ${path.relative(__dirname, sourcePath)}`);

    // 使用clean-css压缩CSS
    // 如果没有安装clean-css，先安装
    try {
      require.resolve("clean-css");
    } catch (e) {
      console.log("安装clean-css...");
      execSync("npm install clean-css --no-save");
    }

    const CleanCSS = require("clean-css");
    const code = fs.readFileSync(sourcePath, "utf8");
    const minified = new CleanCSS().minify(code);

    fs.writeFileSync(targetPath, minified.styles);
  } catch (error) {
    console.error(`处理CSS文件失败: ${sourcePath}`, error);
    // 出错时直接复制原文件
    fs.copyFileSync(sourcePath, targetPath);
  }
}

// 压缩HTML文件
function minifyHTML(sourcePath, targetPath) {
  try {
    console.log(`压缩HTML: ${path.relative(__dirname, sourcePath)}`);

    // 使用html-minifier压缩HTML
    // 如果没有安装html-minifier，先安装
    try {
      require.resolve("html-minifier");
    } catch (e) {
      console.log("安装html-minifier...");
      execSync("npm install html-minifier --no-save");
    }

    const minify = require("html-minifier").minify;
    const code = fs.readFileSync(sourcePath, "utf8");

    const result = minify(code, {
      removeComments: true,
      collapseWhitespace: true,
      minifyJS: true,
      minifyCSS: true,
    });

    fs.writeFileSync(targetPath, result);
  } catch (error) {
    console.error(`处理HTML文件失败: ${sourcePath}`, error);
    // 出错时直接复制原文件
    fs.copyFileSync(sourcePath, targetPath);
  }
}

// 压缩JSON文件
function minifyJSON(sourcePath, targetPath) {
  try {
    console.log(`压缩JSON: ${path.relative(__dirname, sourcePath)}`);

    const code = fs.readFileSync(sourcePath, "utf8");
    const json = JSON.parse(code);
    fs.writeFileSync(targetPath, JSON.stringify(json));
  } catch (error) {
    console.error(`处理JSON文件失败: ${sourcePath}`, error);
    // 出错时直接复制原文件
    fs.copyFileSync(sourcePath, targetPath);
  }
}

// 主函数
function build() {
  console.log("开始打包项目...");
  const startTime = Date.now();

  // 确保dist目录存在
  const distPath = ensureDistDirectory();

  // 需要复制的文件和目录
  const filesToCopy = [
    "manifest.json",
    "assets",
    "background",
    "config",
    "content",
    "options",
    "popup",
    "services",
    "styles",
    "utils",
  ];

  // 复制并压缩文件
  for (const file of filesToCopy) {
    const sourcePath = path.join(__dirname, file);
    const targetPath = path.join(distPath, file);

    if (fs.existsSync(sourcePath)) {
      copyAndMinify(sourcePath, targetPath);
    } else {
      console.warn(`警告: 文件或目录不存在: ${file}`);
    }
  }

  const endTime = Date.now();
  console.log(`打包完成! 耗时: ${(endTime - startTime) / 1000}秒`);
  console.log(`输出目录: ${distPath}`);
}

// 执行构建
build();
