import { showMessage } from "../../../utils/index.js";
import { TimeSaleSettingService } from "../../../services/index.js";

/**
 * 开始按钮管理类
 * 负责处理限时特价设置的开始按钮注册和处理逻辑
 */
export class BeginButtonManager {
  /**
   * 构造函数
   * @param {Object} priceRangeManager - 价格区间管理器实例
   * @param {Object} tempStorage - 本地临时数据存储实例
   * @param {Object} syncClass - 同步类实例
   */
  constructor(priceRangeManager, tempStorage, syncClass) {
    this.priceRangeManager = priceRangeManager;
    this.tempStorage = tempStorage;
    this.syncClass = syncClass;
    this.isRunning = false; // 添加运行状态标志
  }

  /**
   * 注册开始按钮点击事件
   * @param {HTMLElement} container - 包含开始按钮的容器元素
   */
  registerBeginButton(container) {
    const startButton = container.querySelector(".start-btn");
    const stopButton = container.querySelector(".stop-btn");

    // 绑定开始按钮点击事件
    startButton.addEventListener("click", async () => {
      try {
        // 检查是否是付费用户或有免费使用额度
        if (
          this.syncClass.isPaidUser === false &&
          this.syncClass.hasFreeUsage === false
        ) {
          // 导入并显示充值引导弹框
          import("../../components/rechargeModal/rechargeModal.js").then(
            (module) => {
              const { RechargeModal } = module;
              const rechargeModal = RechargeModal.getInstance();
              rechargeModal.show();
            }
          );
          return;
        }

        startButton.style.display = "none"; // 隐藏开始按钮
        stopButton.style.display = "block"; // 显示停止按钮
        this.isRunning = true;
        this.handleBegin();
      } catch (error) {
        startButton.style.display = "block"; // 出错时恢复显示开始按钮
        stopButton.style.display = "none"; // 隐藏停止按钮
        console.error("打开页面失败:", error);
      }
    });

    // 绑定停止按钮点击事件
    stopButton.addEventListener("click", () => {
      try {
        this.handleStop();
      } catch (error) {
        console.error("停止失败:", error);
      }
    });
  }

  /**
   * 处理停止按钮点击事件
   */
  async handleStop() {
    if (!this.isRunning) {
      return;
    }

    this.isRunning = false;
    showMessage("正在停止操作", "warning");

    // 重置按钮状态
    const container = document.getElementById("start-container");
    if (container) {
      const startButton = container.querySelector(".start-btn");
      const stopButton = container.querySelector(".stop-btn");
      if (startButton) startButton.style.display = "block"; // 显示开始按钮
      if (stopButton) stopButton.style.display = "none"; // 隐藏停止按钮
    }
    /*
    // 关闭所有相关标签页
    try {
      // 关闭pop_Goods_Search页面
      chrome.tabs.query({}, (tabs) => {
        for (let i = 0; i < tabs.length; i++) {
          if (tabs[i]?.url?.includes("pop_Goods_Search.aspx") || 
              tabs[i]?.url?.includes("pop_TimeSaleSetting.aspx") ||
              tabs[i]?.url?.includes("ItemDiscountManagement.aspx")) {
            chrome.tabs.remove(tabs[i].id);
          }
        }
      });
      
      showMessage("已停止操作", "success");
    } catch (error) {
      console.error("关闭标签页失败:", error);
      showMessage("停止操作失败", "error");
    }
    */
  }

  /**
   * 处理开始按钮点击事件
   */
  async handleBegin() {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    const that = this;
    // 如果没有填写优惠规则数据不能开始
    if (!this.priceRangeManager.getSaleRuleDatas()) {
      this.isRunning = false;
      const container = document.getElementById("start-container");
      if (container) {
        const startButton = container.querySelector(".start-btn");
        const stopButton = container.querySelector(".stop-btn");
        if (startButton) startButton.style.display = "block"; // 显示开始按钮
        if (stopButton) stopButton.style.display = "none"; // 隐藏停止按钮
      }
      return showMessage("请先设置优惠规则", "error");
    }
    const userId = this.tempStorage.getShopId();
    // 获取店铺商品信息并检查付费状态
    const canContinue = await this.syncClass.getStoreProductsInfo(userId);

    // 如果不能继续（非付费用户且无免费额度），则停止执行
    if (!canContinue) {
      this.isRunning = false;
      const container = document.getElementById("start-container");
      if (container) {
        const startButton = container.querySelector(".start-btn");
        const stopButton = container.querySelector(".stop-btn");
        if (startButton) startButton.style.display = "block"; // 显示开始按钮
        if (stopButton) stopButton.style.display = "none"; // 隐藏停止按钮
      }
      import("../../components/rechargeModal/rechargeModal.js").then(
        (module) => {
          const { RechargeModal } = module;
          const rechargeModal = RechargeModal.getInstance();
          rechargeModal.show();
        }
      );
      return;
    }
    // 获取当前标签页
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });
    if (!tab) {
      console.error("无法获取当前标签页");
      return;
    }

    /**
     * 获取折扣网格数据
     * @param {string} productId - 商品ID
     * @returns {Promise<Array>} - 返回折扣网格数据
     */
    function getDiscountGridData(productId) {
      // 等待元素加载完成
      function waitForElement(selector, maxWaitTime = 10000) {
        return new Promise((resolve, reject) => {
          const startTime = Date.now();

          function checkElement() {
            const element = document.querySelector(selector);
            if (element) {
              resolve(element);
              return;
            }

            if (Date.now() - startTime > maxWaitTime) {
              reject(new Error(`等待元素 ${selector} 超时`));
              return;
            }

            setTimeout(checkElement, 500);
          }

          checkElement();
        });
      }

      // 设置日期范围
      async function setDateRange() {
        // 等待日期输入框加载完成
        const searchDate1 = await waitForElement("#SearchDate1");
        const searchDate2 = await waitForElement("#SearchDate2");

        // 获取上个月1号
        const startDate = new Date();
        startDate.setMonth(startDate.getMonth() - 1);
        startDate.setDate(1);

        // 获取下个月最后一天
        const endDate = new Date();
        endDate.setMonth(endDate.getMonth() + 2);
        endDate.setDate(0);

        // 格式化日期 yyyy-MM-dd
        const formatDate = (date) => {
          const year = date.getFullYear();
          const month = String(date.getMonth() + 1).padStart(2, "0");
          const day = String(date.getDate()).padStart(2, "0");
          return `${year}-${month}-${day}`;
        };

        // 设置日期并触发change事件
        searchDate1.value = formatDate(startDate);
        searchDate2.value = formatDate(endDate);

        // 触发change事件
        searchDate1.dispatchEvent(new Event("change", { bubbles: true }));
        searchDate2.dispatchEvent(new Event("change", { bubbles: true }));
      }

      // 设置搜索条件
      async function setSearchCondition(productId) {
        // 等待搜索类型选择框和输入框加载完成
        const searchType = await waitForElement("#searchType");
        const searchInput = await waitForElement("#txtSearchValue");

        // 选择"商品代码"选项并触发change事件
        searchType.value = "GD_NO";
        searchType.dispatchEvent(new Event("change", { bubbles: true }));

        // 输入商品ID并触发input事件
        searchInput.value = productId;
        searchInput.dispatchEvent(new Event("input", { bubbles: true }));
      }

      // 执行搜索
      async function executeSearch() {
        const searchButton = await waitForElement("#btnSearch");
        searchButton.click();
      }
      // 获取表格数据
      async function getGridData() {
        // 等待表格加载完成
        const discountGrid = await waitForElement(
          "#ctl00_ctl00_MainMaster_MainHolder_DiscountGrid"
        );
        if (!discountGrid) return null;
        const tbody = discountGrid.querySelector(
          "#__grid_DiscountGrid .objbox"
        );
        const rows = tbody.querySelectorAll("tr");
        const gridData = [];

        rows.forEach((row) => {
          const cells = row.querySelectorAll("td");
          if (cells.length > 0) {
            const goodsCode = cells[2]?.textContent?.trim();
            const constdiscountType = cells[3]?.textContent?.trim();
            const goodsName = cells[4]?.textContent?.trim();
            const discountAmount = cells[5]?.textContent?.trim();
            const originalPrice = cells[6]?.textContent?.trim();
            const discountPrice = cells[7]?.textContent?.trim();
            const stock = cells[8]?.textContent?.trim();
            const date = cells[9]?.textContent?.trim();
            const timeSlot = cells[10]?.textContent?.trim();
            if (goodsCode) {
              const cellobj = {
                goodsCode,
                constdiscountType,
                goodsName,
                discountAmount,
                originalPrice,
                discountPrice,
                stock,
                date,
                timeSlot,
              };
              gridData.push(cellobj);
            }
          }
        });

        return gridData;
      }

      // 执行所有操作
      return new Promise(async (resolve, reject) => {
        try {
          // 1. 等待5秒确保页面基本加载完成
          await new Promise((resolve) => setTimeout(resolve, 5000));
          // 2. 设置日期范围
          await setDateRange();

          // 3. 设置搜索条件
          await setSearchCondition(productId);

          // 4. 点击搜索按钮
          await executeSearch();

          // 5. 等待数据加载完成后获取数据
          await new Promise((resolve) => setTimeout(resolve, 3000));
          const data = await getGridData();
          resolve(data);
        } catch (error) {
          console.error("获取数据失败:", error);
          reject(error);
        }
      });
    }

    // 打开限时特价页面
    const url =
      "https://qsm.qoo10.jp/GMKT.INC.Gsm.Web/Goods/ItemDiscountManagement.aspx";
    await chrome.tabs.update(tab.id, { url });
    // 等待页面加载完成后获取数据
    chrome.tabs.onUpdated.addListener(function discountManagementListener(
      tabId,
      info
    ) {
      if (tabId === tab.id && info.status === "complete") {
        chrome.tabs.onUpdated.removeListener(discountManagementListener);
        // 如果已经停止运行，则不继续执行
        if (!that.isRunning) {
          return;
        }
        const productId = that.priceRangeManager.getCurrentProductId();
        if (!productId) {
          return showMessage("无法获取商品ID", "error");
        }
        showMessage(`执行商品${productId}`, "success");
        chrome.scripting.executeScript(
          {
            target: { tabId: tab.id },
            function: getDiscountGridData,
            args: [productId],
          },
          (res) => {
            if (!that.isRunning) {
              return;
            }
            if (!res || !Array.isArray(res)) {
              showMessage("获取数据失败", "error");
              return;
            }
            const { result } = res[0];
            if (Array.isArray(result) && result.length > 0) {
              // 执行下次操作-当前商品已经执行过
              that.completeOneExecution();
            } else {
              that.handlePopupAutomation();
            }
          }
        );
      }
    });
  }

  /**
   * 处理弹出自动化
   */
  async handlePopupAutomation() {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    const that = this;
    // 获取当前标签页
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });
    // 先关闭之前的pop_Goods_Search页面
    chrome.tabs.query({}, (tabs) => {
      for (let i = 0; i < tabs.length; i++) {
        if (tabs[i]?.url?.includes("pop_Goods_Search.aspx")) {
          chrome.tabs.remove(tabs[i].id);
        }
      }
    });
    // 打开pop_Goods_Search弹窗
    chrome.scripting.executeScript({
      target: { tabId: tab.id },
      function: function () {
        const url =
          "https://qsm.qoo10.jp/GMKT.INC.Gsm.Web/Popup/Goods/pop_Goods_Search.aspx?callback=SetGoodsInfo";
        window.open(
          url,
          "find_goods",
          "top=100,left=100,Width=610, Height=520"
        );
      },
    });
    let recordTabId;
    chrome.tabs.onUpdated.addListener(function goodsInfoListener(tabId, info) {
      if (
        info?.title?.includes("Qoo10 - QSM") ||
        info?.title?.includes("Gマーケット - QSM")
      ) {
        recordTabId = tabId;
      }
      if (info.status === "complete" && tabId === recordTabId) {
        chrome.tabs.onUpdated.removeListener(goodsInfoListener);
        // 如果已经停止运行，则不继续执行
        if (!that.isRunning) {
          return;
        }
        that.handleAutoPage(tabId).then(() => {
          if (that.isRunning) {
            that.handleDiscountValue();
          }
        });
      }
    });
  }

  /**
   * 处理自动页面
   * @param {number} tabId - 标签页ID
   * @returns {Promise<boolean>} - 返回处理结果
   */
  async handleAutoPage(tabId) {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return false;
    }

    showMessage("选择商品", "success");
    const productId = this.priceRangeManager.getCurrentProductId();
    await chrome.scripting.executeScript({
      target: { tabId },
      function: async (productId) => {
        // 填充ID
        function fillID(productId) {
          const input = document.getElementById("txt_serach_value");
          if (input) {
            input.value = productId;
            // 触发输入框的change事件
            const event = new Event("change");
            input.dispatchEvent(event);
          }
        }
        // 触发搜索
        function searchBtn() {
          const button = document.getElementById("btn_search");
          if (button) {
            button.click();
          }
        }
        // 选中数据
        async function selectData() {
          const selectId = ".objbox>table>tbody>tr:nth-child(2)>td";
          await waitForDataToLoad(selectId); // 使用轮询等待数据加载完成
          const checkboxes = document.querySelector(selectId);
          if (checkboxes) {
            checkboxes.click();
          }
        }
        // 确认选中
        function confirmSelect() {
          return new Promise((resolve, reject) => {
            const button = document.getElementById("btn_select");
            const btnColseDom = document.querySelector(
              "#div_contents #btn_close"
            );
            setTimeout(() => {
              if (button) {
                button.click();
                if (btnColseDom) {
                  setTimeout(() => {
                    btnColseDom.click();
                  }, 1000);
                }
                resolve();
              }
            }, 1000);
          });
        }
        // 等待数据加载完成
        async function waitForDataToLoad(selectId, maxWaitTime = 60000) {
          return new Promise((resolve, reject) => {
            const startTime = Date.now();
            const interval = setInterval(() => {
              const dataRows = document.querySelector(selectId);
              const noDataDom = document.querySelector(
                "#Grid_Option #__grid_ListGoodsOpt_nodata"
              );
              const noDataDomIsVisible = noDataDom?.style?.display === "block";
              if (dataRows || noDataDomIsVisible) {
                clearInterval(interval);
                resolve();
              } else if (Date.now() - startTime > maxWaitTime) {
                clearInterval(interval);
                reject(new Error(`等待元素 ${selectId} 超时`));
              }
            }, 500);
          });
        }

        fillID(productId);
        searchBtn();
        await selectData();
        await confirmSelect();
        return true;
      },
      args: [productId],
    });
    return true;
  }

  /**
   * 处理折扣值
   */
  async handleDiscountValue() {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    showMessage("填写优惠数据", "success");
    // 获取当前标签页
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });
    // 获取 priceRangeManager 的数据
    const saleRuleDatas = this.priceRangeManager.getSaleRuleDatas();
    chrome.scripting
      .executeScript({
        target: { tabId: tab.id },
        args: [saleRuleDatas], // 将 saleRuleDatas 作为参数传递
        function: function (saleRuleDatas) {
          // 获取价格
          const sellPriceDom = document.getElementById("sell_price");
          if (sellPriceDom) {
            const sellPriceValue = parseFloat(sellPriceDom.value);
            const priceRules = saleRuleDatas.priceRules;
            const perPersonLimit = saleRuleDatas.perPersonLimit;
            const totalLimit = saleRuleDatas.totalLimit;
            // 匹配价格区间并获取优惠金额
            let discountAmount = 0;
            for (const rule of priceRules) {
              const startPrice = parseFloat(rule.startPrice);
              const endPrice = parseFloat(rule.endPrice);
              if (sellPriceValue >= startPrice && sellPriceValue <= endPrice) {
                discountAmount = parseFloat(rule.discountAmount);
                break;
              }
            }
            // 获取優惠金額
            const discountDom = document.getElementById("s_cost_price");
            if (discountDom) {
              discountDom.value = discountAmount;
            }

            // 数量制限
            const discountQtyDom = document.getElementById("txtDiscountQtyCnt");
            if (discountQtyDom) {
              discountQtyDom.value = parseFloat(totalLimit);
            }
            // 每人：ID：txtOrderLimitCnt
            const txtOrderLimitCntDom =
              document.getElementById("txtOrderLimitCnt");
            if (txtOrderLimitCntDom) {
              txtOrderLimitCntDom.value = parseFloat(perPersonLimit);
            }
          }
          return true;
        },
      })
      .then(() => {
        if (this.isRunning) {
          setTimeout(() => {
            this.handleTimeRange();
          }, 1000);
        }
      });
  }

  /**
   * 处理时间范围
   */
  async handleTimeRange() {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    const that = this;
    showMessage("设置期间日期", "success");
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });

    chrome.scripting.executeScript({
      target: { tabId: tab.id },
      function: function () {
        const start_timezone = document.getElementById("sel_timeZone").value;
        const start_dt = document.getElementById("S_SDAY").value;
        const url = `https://qsm.qoo10.jp/GMKT.INC.Gsm.Web/Popup/Goods/pop_TimeSaleSetting.aspx?s_dt=${start_dt}&timezone=${start_timezone}`;
        window.open(
          url,
          "TimeSaleSetting",
          "top=300,left=500,width=800,height=470,scrollbars=no"
        );
        return true;
      },
    });
    let recordTabId;
    chrome.tabs.onUpdated.addListener(function timeSaleSettingListener(
      tabId,
      info
    ) {
      if (
        info?.title?.includes("QSM (Qoo10 Sales Manager)") ||
        info?.title?.includes("QSM (Qoo10 Sales Manager)")
      ) {
        recordTabId = tabId;
      }
      if (info.status === "complete" && tabId === recordTabId) {
        chrome.tabs.onUpdated.removeListener(timeSaleSettingListener);
        // 如果已经停止运行，则不继续执行
        if (!that.isRunning) {
          return;
        }
        that.setTimeRange(tabId);
      }
    });
  }

  /**
   * 设置时间范围
   * @param {number} tabId - 标签页ID
   */
  setTimeRange(tabId) {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    /**
     * 处理时间范围脚本
     */
    function handleTimeRangeScript() {
      return new Promise((resolve, reject) => {
        function autoConfirm() {
          // 保存原始confirm引用
          window.__nativeConfirm = window.confirm;
          window.__nativeAlert = window.alert;

          const newConfirm = function () {
            setTimeout(() => {
              window.confirm = window.__nativeConfirm;
            });
            return true;
          };
          const newAlert = function () {
            setTimeout(() => {
              window.alert = window.__nativeAlert;
            });
            return true;
          };
          // 重写confirm
          window.confirm = newConfirm;
          window.alert = newAlert;
        }
        autoConfirm();

        function getNextMonthPrevDay(dateStr) {
          const parts = dateStr.split("/");
          const year = parseInt(parts[0], 10);
          const month = parseInt(parts[1], 10) - 1; // 月份转为0-based
          const day = parseInt(parts[2], 10);
          const date = new Date(year, month, day);

          const currentMonth = date.getMonth();
          date.setMonth(currentMonth + 1); // 先加一个月

          // 处理月份溢出（如1月31日加一个月后变成3月）
          if (date.getMonth() !== (currentMonth + 1) % 12) {
            // 获取目标月份的最后一天（currentMonth+2月的第0天）
            date.setTime(new Date(year, currentMonth + 2, 0).getTime());
          }

          date.setDate(date.getDate() - 1); // 再减一天

          // 格式化为yyyy/mm/dd
          const y = date.getFullYear();
          const m = String(date.getMonth() + 1).padStart(2, "0");
          const d = String(date.getDate()).padStart(2, "0");
          return `${y}/${m}/${d}`;
        }

        // 设定一个连续的时期
        const checkContinuePeriod = document.getElementById(
          "chk_continue_period"
        );
        if (checkContinuePeriod) {
          checkContinuePeriod.click();
        }
        // 设置结束时间
        const endDate = document.getElementById("End_Date");
        const startDate = document.getElementById("Start_Date");
        if (endDate && startDate) {
          const newEndDate = getNextMonthPrevDay(startDate.value);
          endDate.value = newEndDate;
        }
        // 点击添加按钮
        const btnAdd = document.getElementById("btnAdd");
        if (btnAdd) {
          btnAdd.click();
        }
        // 点击保存
        const btnSave = document.getElementById("btnSave");
        if (btnSave) {
          setTimeout(() => {
            btnSave.click();
            resolve();
          }, 1000);
        } else {
          reject(new Error("btnSave is null"));
        }
        // 取消confirm代理
        function clearConfirmProxy() {
          window.confirm = window.__nativeConfirm;
          window.alert = window.__nativeAlert;
        }
        clearConfirmProxy();
      });
    }

    chrome.scripting.executeScript(
      {
        target: { tabId },
        function: handleTimeRangeScript,
      },
      () => {
        if (this.isRunning) {
          this.handleSaveAndInterceptConfirm();
        }
      }
    );
  }

  /**
   * 处理保存并拦截确认
   */
  async handleSaveAndInterceptConfirm() {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    showMessage("保存中，勿动", "success", 20000);
    const [tab] = await chrome.tabs.query({
      active: true,
      currentWindow: true,
    });

    chrome.scripting.executeScript(
      {
        target: { tabId: tab.id },
        world: "MAIN", // 注入到主环境
        func: () => {
          return new Promise((resolve) => {
            // 保存原始confirm引用
            window.__nativeConfirm = window.confirm;
            window.__nativeAlert = window.alert;

            const newConfirm = function () {
              setTimeout(() => {
                window.confirm = window.__nativeConfirm;
              });
              return true;
            };
            const newAlert = function () {
              setTimeout(() => {
                window.alert = window.__nativeAlert;
              });
              resolve(true);
              return true;
            };
            // 重写confirm
            window.confirm = newConfirm;
            window.alert = newAlert;
            // 获取保存按钮
            const btnDiscountApply =
              document.getElementById("btnDiscountApply");
            if (btnDiscountApply) {
              btnDiscountApply.click();
            }
          });
        },
      },
      (res) => {
        if (!this.isRunning) {
          return;
        }
        if (!res || !Array.isArray(res)) {
          showMessage("执行失败", "error");
          return;
        }
        this.completeOneExecution();
      }
    );
  }

  /**
   * 完成一次执行
   */
  completeOneExecution() {
    // 如果已经停止运行，则不继续执行
    if (!this.isRunning) {
      return;
    }

    const productId = this.priceRangeManager.getCurrentProductId();
    const userId = this.tempStorage.getShopId();
    const params = {
      platformShopId: userId,
      productCode: productId,
      success: true,
      message: "设置成功",
    };
    TimeSaleSettingService.updateShopDiscountCallback(params).then(() => {
      showMessage("设置成功", "success");
      setTimeout(() => {
        if (this.isRunning) {
          this.handleBegin();
        }
      }, 2000);
    });
  }
}
